//*************************************************************************************************
//
//	Description:
//		grass_billboard.fx - Shader for billboarded rendering of grass tufts.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship			20/06/2008  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

// This shader doesn't benefit from soft shadows
#define FORCE_HARD_SHADOWS
#define _SSAO_READY_
#include "stddefs.fxh"
#include "specialisation_globals.fxh"



//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define AUTOGRASS
#endif

#define NUM_TEXTURE_SUBPAGES				2.0f
#define OO_NUM_TEXTURE_SUBPAGES		0.5f


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;

float4x4 viewI : ViewI
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif



//
// Transforms
//

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
	bool export = false;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_
// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Second UV holds the billboarding pivot offsets
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
	bool use3dMapping = true;
> = 0;
#endif


// Automatic grass or not
SPECIALISATION_PARAM( autograss, "Autograss?", "AUTOGRASS" )	// TRUE if we're using this material for automatic placement

#if defined( _3DSMAX_ ) || defined( AUTOGRASS )

// fade distances
float2 fadeDistances
<
	bool appEdit = true;
> = { 0.0f, 200.0f };

float globalFade
<
	bool appEdit = true;
> = 1.0f;

// texture subpage sizes
uniform float4 subPageWidths
<
	bool appEdit = true;
>;

uniform float4 subPageHeights
<
	bool appEdit = true;
>;

// box world size
float boxWorldCoordsSize
<
	bool appEdit = true;
> = 16.0f;


#endif



//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
#ifdef _3DSMAX_
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 billboardPivot : TEXCOORD1;
	float4 randoms  : TEXCOORD2;													// Random numbers
#else

// Autograss system provides very different input params
#if defined( AUTOGRASS )
	float3 billboardPivot : POSITION;											// Parametric position (top view) of pivot point (y always 0.0)
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float4 randoms  : TEXCOORD1;													// Random numbers. Consistent per quad.
	float ypos			: TEXCOORD2;													// World coords Y position of pivot
	float4 controls : TEXCOORD3;													// Control values. X : density, Y : std height, Z : WC normal X component, W : WC normal Z component
#else
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float3 billboardPivot : TEXCOORD1;
#endif		// AUTOGRASS

#endif	// 3DSMAX
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD2;												// Vertex colour
	float2 texCoord	: TEXCOORD0;												// UV coords for texture channel 0
	float4 eye				  : TEXCOORD1;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 3 )
};




struct ZPRIMEDOF_VSINPUT
{
// Autograss system provides very different input params
#if defined( AUTOGRASS )
	float3 billboardPivot : POSITION;											// Parametric position (top view) of pivot point (y always 0.0)
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float4 randoms  : TEXCOORD1;													// Random numbers. Consistent per quad.
	float ypos			: TEXCOORD2;													// World coords Y position of pivot
	float4 controls : TEXCOORD3;													// Control values. X : density, Y : std height, Z : WC normal X component, W : WC normal Z component
#else
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float3 billboardPivot : TEXCOORD1;
#endif		// AUTOGRASS
};



struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord		: TEXCOORD0;
	float4 coords				: TEXCOORD1;
	float alpha					: TEXCOORD2;
};


//-----------------------------------------------------------------------
//
// Functions
//

float3 LockBillboardingAxis( float3 _result, float3 _billboardLockAxis, float3 _billboardPivot, float4x4 _view )
{
#ifdef _3DSMAX_
	float3 upAxis = mul( _billboardLockAxis, (float3x3)_view );
	float3 rightAxis = normalize( cross( upAxis, _result ) );
	float3 forwardAxis = cross( rightAxis, upAxis );	
	return ( _billboardPivot.x * rightAxis + _billboardPivot.y * upAxis + _billboardPivot.z * forwardAxis );
	
#else
	float3 upAxis = mul( _billboardLockAxis, (float3x3)_view );	
	float3 rightAxis = normalize( cross( upAxis, _result ) );
	float3 forwardAxis = cross( rightAxis, upAxis );	

	return ( _billboardPivot.x * rightAxis + _billboardPivot.y * upAxis + _billboardPivot.z * forwardAxis );
#endif	

	return _result;
}

// Cut down version of the usual function, as we're always Y-locking grass tufts
float3 DoBillboardingZLock( float3 _inputPosition, float3 _billboardPivot, float4x4 _worldview, float4x4 _view )
{
#ifdef _3DSMAX_
	_billboardPivot.y += 1;
	float3 pivotPoint = _inputPosition - _billboardPivot;
#else
	float3 pivotPoint = _inputPosition - _billboardPivot.xzy;
#endif

	float3 result = mul( float4( pivotPoint, 1.0f ), _worldview ).xyz;

#ifdef _3DSMAX_
	result += LockBillboardingAxis( result, float3( 0, 0, 1 ), _billboardPivot, _view );
#else
	result += LockBillboardingAxis( result, float3( 0, 1, 0 ), _billboardPivot, _view );
#endif

	return result;
}


#if defined( _3DSMAX_ ) || defined( AUTOGRASS )

#if defined( _PS3_ )
// The PS3 version requires this atrocious hack to get around limitations in its awful shader compiler.
uniform const float4 twatty[4]={{1, 0, 0, 0}, {0,1, 0, 0}, {0,0,1,0}, {0,0,0,1}};
#endif


float2 CalculateSubTextureWorldSize( int _index )
{

#if defined( _PS3_ )
	float2 result;
	result.x = dot( twatty[ _index ], subPageWidths );
	result.y = dot( twatty[ _index ], subPageHeights );

	return result;

#else

	float2 result;

	result.x = subPageWidths[ _index ];
	result.y = subPageHeights[ _index ];

	return result;
#endif
}

#endif


bool DecideToRender( float _density, float4 _randoms )
{
	return ( _randoms.w < _density );
}


int ChooseTexturePage( float _height, float _maxPages, float4 _randoms, float _noise, float _variance )
{
/*
	float distribution = _randoms.y * _variance;

	return saturate( distribution ) * ( _maxPages - 0.01f );
*/
	return _randoms.y * ( _maxPages - 0.01f );
}



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT GrassVertexShader( VSINPUT _input )
{
	VSOUTPUT _output = (VSOUTPUT)0;

	// Init output colour to opaque white
	_output.colour = float4( 1.0f, 1.0f, 1.0f, 1.0f );

	// Autograss calculations
	float3 viewPos = float3( 0.0f, 0.0f, 0.0f );
	float4 worldPos = float4( 0.0f, 0.0f, 0.0f, 1.0f );
	float3 normal = float3( 0.0f, 1.0f, 0.0f );
	bool renderMe = true;

#if defined( AUTOGRASS )
		// Decide if the density means this quad should be drawn or not
		renderMe = DecideToRender( _input.controls.x, _input.randoms );
		if ( renderMe )
		{
			// Get local-coords pivot position from the inputs
			float4 localPivotPos = float4( _input.billboardPivot.x, _input.ypos, _input.billboardPivot.z, 1.0f );

			// Calculate world-coords pivot position (shortcut as we know the input xform is always just a translation which contains a y coord which we need to ignore)
			// This is convoluted, but necessary to force the engine code to apply the appropriate bounding volume.
			float4 worldPivotPos = float4( localPivotPos.x + world[ 3 ].x, localPivotPos.y, localPivotPos.z + world[ 3 ].z, 1.0f );

			// Calculate the normal vector from the control data
			normal.x = _input.controls.z;
			normal.z = _input.controls.w;
			normal.y = 1.0f - sqrt( normal.x * normal.x + normal.z * normal.z );

			// Calculate some seed values of differing variance
//			float seedVariesEverywhere = frac( dot( ( float2( _input.billboardPivot.x, _input.billboardPivot.z ) / boxWorldCoordsSize ) + _input.randoms.xz, 1.0f ) ); 
			float seedVariesEverywhere = _input.randoms.x;

			// Choose the texture page which will be used
			int texturePage = ChooseTexturePage( _input.controls.y, NUM_TEXTURE_SUBPAGES, _input.randoms, seedVariesEverywhere, 1.0f );
			float2 quadSize = CalculateSubTextureWorldSize( texturePage ) * ( ( seedVariesEverywhere * 0.4f ) + 0.8f );

			// For now, arbitrary billboarding
			float4 pivotInView = mul( worldPivotPos, view );
			float3 cornerOffset = float3( ( _input.texCoord.x - 0.5f ) * quadSize.x, ( 0.75f - _input.texCoord.y ) * quadSize.y, 0.0f );
			viewPos = pivotInView.xyz + cornerOffset;

			worldPos = mul( float4( viewPos, 1.0f ), viewI );

			_output.texCoord = float2( ( ( _input.texCoord.x + texturePage ) * OO_NUM_TEXTURE_SUBPAGES), _input.texCoord.y );

			// Calculate distance fade (gradual near, gradual far)
			float distance = sqrt( dot( pivotInView.xyz, pivotInView.xyz ) );
//			float transModifier = 1.0f - saturate( distance / fadeDistances.y );
//			transModifier *= saturate( distance - max( 0.0f, fadeDistances.x - 1.0f ) );
			float transModifier = saturate( distance - max( 0.0f, fadeDistances.x - 1.0f ) );
			transModifier *= globalFade;

			_output.colour.a *= transModifier;

			if ( transModifier < 0.05f )
			{
				renderMe = false;
			}
		}

#else
		float4x4	worldview = mul( world, view );

	_output.texCoord = _input.texCoord;
		viewPos = DoBillboardingZLock( _input.position, _input.billboardPivot.xyz, worldview, view );
#endif

	// If we've decided to render this quad
	if ( renderMe )
	{
		// Calculate clip-space position of the vertex
		_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

		// Calculate world-space vector to the eye (approximate only; the shading on these is so minor, we don't really need to interpolate a vector)
	#ifdef _3DSMAX_
		float3 worldEyeVec = viewI[ 3 ] - worldPos.xyz;
	#else
		float3 worldEyeVec = worldCameraPos - worldPos.xyz;
	#endif
		_output.eye = float4( worldEyeVec, 0.0f );

		// In the engine, specialisations which have no specular defined need to declare these constants
		// as the lighting macros at the end need them, and they're not declared anywhere else.
		float globalSpecularFactorValue = 0.0f;
		float minSpecPowerValue = 0.0f;
		float maxSpecPowerValue = 0.0f;

		// Do lighting
		DO_VS_LIGHTING_CALCULATIONS
	}

	// Decided not to render the quad
	else
	{
		// So output blank values & place the projected vertex offscreen
		_output.position.w = -1.0f;
	}

	return _output;
}



ZPRIMEDOF_VSOUTPUT GrassZPrimeDOFVertexShader( ZPRIMEDOF_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;
	_output.position = float4( 100.0f, 0.0f, 0.0f, 0.0f );
	_output.texCoord = float2( 0.0f, 0.0f );
	_output.coords = _output.position;

	// Init output alpha to opaque
	_output.alpha = 1.0f;

	// Autograss calculations
	float3 viewPos = float3( 0.0f, 0.0f, 0.0f );
	float3 normal = float3( 0.0f, 1.0f, 0.0f );
	bool renderMe = true;

#if defined( AUTOGRASS )
		// Decide if the density means this quad should be drawn or not
		renderMe = DecideToRender( _input.controls.x, _input.randoms );
		if ( renderMe )
		{
			// Get local-coords pivot position from the inputs
			float4 localPivotPos = float4( _input.billboardPivot.x, _input.ypos, _input.billboardPivot.z, 1.0f );

			// Calculate world-coords pivot position (shortcut as we know the input xform is always just a translation which contains a y coord which we need to ignore)
			// This is convoluted, but necessary to force the engine code to apply the appropriate bounding volume.
			float4 worldPivotPos = float4( localPivotPos.x + world[ 3 ].x, localPivotPos.y, localPivotPos.z + world[ 3 ].z, 1.0f );

			// Calculate the normal vector from the control data
			normal.x = _input.controls.z;
			normal.z = _input.controls.w;
			normal.y = 1.0f - sqrt( normal.x * normal.x + normal.z * normal.z );

			// Calculate some seed values of differing variance
//			float seedVariesEverywhere = frac( dot( ( float2( _input.billboardPivot.x, _input.billboardPivot.z ) / boxWorldCoordsSize ) + _input.randoms.xz, 1.0f ) ); 
			float seedVariesEverywhere = _input.randoms.x;

			// Choose the texture page which will be used
			int texturePage = ChooseTexturePage( _input.controls.y, NUM_TEXTURE_SUBPAGES, _input.randoms, seedVariesEverywhere, 1.0f );
			float2 quadSize = CalculateSubTextureWorldSize( texturePage ) * ( ( seedVariesEverywhere * 0.2f ) + 1.0f );

			// For now, arbitrary billboarding
			float4 pivotInView = mul( worldPivotPos, view );
			float3 cornerOffset = float3( ( _input.texCoord.x - 0.5f ) * quadSize.x, ( 0.75f - _input.texCoord.y ) * quadSize.y, 0.0f );
			viewPos = pivotInView.xyz + cornerOffset;

			_output.texCoord = float2( ( ( _input.texCoord.x + texturePage ) * OO_NUM_TEXTURE_SUBPAGES ), _input.texCoord.y );

			// Calculate distance fade (gradual near, gradual far)
			float distance = sqrt( dot( pivotInView.xyz, pivotInView.xyz ) );
//			float transModifier = 1.0f - saturate( distance / fadeDistances.y );
//			transModifier *= saturate( distance - max( 0.0f, fadeDistances.x - 1.0f ) );
			float transModifier = saturate( distance - max( 0.0f, fadeDistances.x - 1.0f ) );
			transModifier *= globalFade;

			_output.alpha *= transModifier;

			if ( transModifier < 0.05f )
			{
				renderMe = false;
			}
		}

#else
		float4x4	worldview = mul( world, view );

		_output.texCoord = _input.texCoord;
		viewPos = DoBillboardingZLock( _input.position, _input.billboardPivot.xyz, worldview, view );
#endif

	// If we've decided to render this quad
	if ( renderMe )
	{
		// Calculate clip-space position of the vertex
		_output.position = mul( float4( viewPos, 1.0f ), projMatrix );
		_output.coords = _output.position;
	}

	// Decided not to render the quad
	else
	{
		// So place the projected vertex offscreen
		_output.position.w = -1.0f;
	}

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD2;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 eye				: TEXCOORD1;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 3 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD2;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 eye				: TEXCOORD1_centroid;								// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 3 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};


//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT GrassFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float4 diffuseTexColour;

	// Read diffuse colour
	diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float4 specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 0.0f;
	float maxSpecPowerValue = 0.0f;

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

  // Normalise interpolated vectors
	float3 TSnormal = float3( 0.0f, 1.0f, 0.0f );
	float3 normal = TSnormal;
  float3 eye = normalize( _input.eye.xyz );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}



PSOUTPUT GrassZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.alpha;

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );

	return output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Grass
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	bool isBillboard = true;

	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Grass";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Grass_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 128;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 128;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GrassVertexShader();
		PixelShader = compile sce_fp_rsx GrassFragmentShader();
#else
		VertexShader = compile vs_3_0 GrassVertexShader();
		PixelShader = compile ps_3_0 GrassFragmentShader();
#endif
	}
}


technique _Grass_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 128);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx GrassZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx GrassZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 128;
	    AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 GrassZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 GrassZPrimeDOFFragmentShader();
#endif
	}
}
